
import {
    userDb,
    createDefaultSubscription,
    checkEmailLimit,
    emailUsageDb,
    subscriptionPlansDb,
    userSubscriptionsDb
} from '../src/lib/database.js';
import { neon } from '@neondatabase/serverless';

// Mock process.env if needed or rely on dotenv which we will run with
if (!process.env.DATABASE_URL) {
    console.error('DATABASE_URL is not set. Please run with dotenv/config.');
    process.exit(1);
}

const sql = neon(process.env.DATABASE_URL);

async function runTest() {
    console.log('🧪 Starting Email Generation Limit Test...');

    // 1. Create a test user
    const uniqueId = Math.floor(Math.random() * 100000);
    const testUser = {
        name: `Test User ${uniqueId}`,
        email: `test_limit_${uniqueId}@example.com`,
        company: 'Test Corp',
        job_title: 'Tester'
    };

    try {
        console.log(`\n👤 Creating test user: ${testUser.email}...`);
        const { lastInsertRowid: userId } = await userDb.create(testUser);
        console.log(`   User created with ID: ${userId}`);

        // 2. Create default subscription
        console.log(`\n🎫 Creating default subscription...`);
        await createDefaultSubscription(userId);
        const subscription = await userSubscriptionsDb.getCurrent(userId);
        console.log(`   Subscription created: ${subscription.plan_name} (Limit: ${subscription.generation_limit})`);

        // 3. Check initial limit status
        console.log(`\n🔍 Checking initial limit status...`);
        let limitCheck = await checkEmailLimit(userId, 'generation');
        console.log(`   Allowed: ${limitCheck.allowed}, Remaining: ${limitCheck.remaining}`);

        if (!limitCheck.allowed) {
            throw new Error('User should be allowed to generate emails initially.');
        }

        // 4. Consume all generations
        console.log(`\n📉 Consuming all generations...`);
        const limit = subscription.generation_limit;

        for (let i = 0; i < limit; i++) {
            process.stdout.write(`   Generating email ${i + 1}/${limit}... `);
            await emailUsageDb.incrementGeneration(userId);

            // Check limit after each increment (optional, but good for verification)
            const check = await checkEmailLimit(userId, 'generation');
            console.log(`Remaining: ${check.remaining}`);
        }

        // 5. Verify limit reached
        console.log(`\n🚫 Verifying limit reached...`);
        limitCheck = await checkEmailLimit(userId, 'generation');
        console.log(`   Allowed: ${limitCheck.allowed}, Remaining: ${limitCheck.remaining}, Reason: ${limitCheck.reason}`);

        if (limitCheck.allowed) {
            throw new Error('User should NOT be allowed to generate emails after hitting limit.');
        } else {
            console.log('   ✅ CORRECT: Limit enforcement working as expected.');
        }

        // Cleanup (Optional)
        // console.log('\n🧹 Cleaning up test user...');
        // await sql`DELETE FROM users WHERE id = ${userId}`;

        console.log('\n✅ TEST COMPLETED SUCCESSFULLY');
        process.exit(0);

    } catch (error) {
        console.error('\n❌ TEST FAILED:', error);
        process.exit(1);
    }
}

runTest();
